##########################################################################
# TPM 2.0 Sample TXT Provisioning Script  
# Copyright Intel Corporation 2015
# Last update: Oct 23, 2015
#   Dec 2014 added algorithm agility
#   May 2015 added platform agility and support for new NV Index Handles
#	OCT 2015 Added additional checks
#   Aug 2017 Added ability to delete PS using alternative policy
##########################################################################
##########################################################################
# Usage:                                                                 #
#     Tpm2TxtProv.nsh   <alg>   <fileNamePrefix>   [-o]                  #
#  Where:                                                                #
#      <alg> is  a supported hash algorithm                              #
#         Such as SHA1, SHA256, SHA384, SHA512, SM3                      #
#         <alg> will be substituted for %1 in Tpm2ProvTool command lines #
#      <fileNamePrefix> is the file name prefix you used to create the   #
#         def files - typically a project name or platform name          #
#        (e.g., MyFirst; Example; Purley)                                #
#         <fileNamePrefix> will be substituted for %2 in command lines   #
#         For example %2Ps%1.iDef --> ExamplePsSha256.iDef               #
#                     %2Ps%1.iDef --> MyFirstPsSha384.iDef               #
#      "-o" is an optional parameter, when specified, uses the original  #
#         AUX index handle(0x01800003) instead of the new (0x01C10102)   #
##########################################################################

echo -OFF 
# Copyrights:
# You may copy, distribute, and make derivatives of this script under the 
# following provisions:
#  You must not remove or modify the Intel copyright notice nor these requirements
#  The script must only be used to provision a TPM for use with Intel Technologies
#    For any other use you must get written permission from Intel Corporation
 
##########################################################################
# This script will:
# Check if the PS NV Index already exists
# - if PS Index does not exist the script will create and write the PS Index 
# - if PS does exist, - checks the PS attributes and NV Data
#   -- if wrong attributes, - deletes the index, recreates it, & writes it
#   -- if data is incorrect, - re-writes the NV data
# Check if the AUX Index already exists
# - if AUX does not exist, creates the AUX Index 
# - If AUX does exist, it checks the index's attributes 
#   -- if wrong attributes - deletes the index, recreates it, 

##########################################################################
# Here is a list of files that are needed
## //User Modified Files -----------[example for %1=SHA256, %2=MyFirst]
## %2SecretOR%1.pDef                  MyFirstSecretOrSha256.pDef
## %2PsFinalOR%1.pDef                 MyFirstPsFinalORSha256.hash
## %2Ps%1.iDef                        MyFirstPsSha256.iDef
## Intel Provided Files -----------------------------------------------------
## Aux%1.iDef                         AuxSha256.iDef
## Aux-o%1.iDef                       Aux-oSha256.iDef
## AuxPolicyOR%1.pDef                 AuxPolicyORSha256.pDef
## EmptyAuthPwSession.sDef
## getsec64.efi    -- part of Intel TXT Toolkit  
## Locality3&4.pDef
## Policy%1.sDef                      PolicySha256.sDef
## PsDelAux.iDef
## PsDelAux-o.iDef
## UndefinespaceSpecial.pDef
##########################################################################
# This script echoes progress markers to the screen and also to
#   a log file (Tpm2TxtProv.log) which also captures provisioning details 
#
#  Note that there are commands indicated by "# For testing on NEX"  
#    These never get executed when the script is run in the EFI environment
#     - they are for ease of manually testing using the NSH Exerciser
#     - Those lines may be deleted  
##########################################################################

#Check that Alg parameter is present
if ZZ%1  == ZZ   then
	echo Algorithm parameter missing
	echo Syntax: %0 <alg> <PlatformName>
	echo   Examples: 
	echo     %0 SHA256 Example  
	echo     %0 SHA384 MyFirst 
	echo     %0 SHA512 Grantley 
	echo     %0 SM3 Dragonfly 
	goto EOF
endif
# Check if alg parameter supported
if NOT EXIST Policy%1.sDef then  
	echo Algorithm not supported - Policy%1.sDef not found
	goto EOF
endif
# Check if platform parameter supported
if NOT EXIST %2Ps%1.iDef then  
	echo Platform/Project Name not found
	goto EOF
endif
# Check if Param3 is correct
if NOT EXIST Aux%3%1.iDef then  
	echo Aux%3%1.iDef not found
	goto EOF
endif

# Now copy this file to the error log
echo TPM2 provisioning Script %1 %2 %3 > Tpm2TxtProv.log
if EXIST %0     then
	type %0     >> Tpm2TxtProv.log
endif
if EXIST %0.nsh then 
	type %0.nsh >> Tpm2TxtProv.log
endif

goto START
##########################################################################
# #####For testing on NEX #####
# The following commands do not need to be sent to a TPM on a 
#   target platform because BIOS starts the TPM  
#   -- rather they are here for testing on a simulator or TTV
#   --   using the NSH Exerciser
#
#  On Simulator start-up or after inserting a TPM in the TTV
Tpm2ProvTool  PowerUp      # for testing on NEX
Tpm2ProvTool  StartUpClear # for testing on NEX
#
# Before removing a TPM from the TTV
Tpm2ProvTool ShutdownClear # for testing on NEX
Tpm2ProvTool PowerDown     # for testing on NEX
##########################################################################

:START
echo ****** Provisioning NV Indexes *****
# Uses Session 0 for AuthPolicy
# Uses Session 1 for Empty AuthValue PW session
# Note that session 0 uses "Don't Continue" so no need to flush on successful use

# This script requires that PlatformAuth be EMPTY in order to create NV Indexes
echo If PlatformAuth is not EMPTY, then first run ResetPlatformAuth.nsh %1 %2

# Start a PW Session to use for PhAuth and Index Read Auth
echo **** Start PW Session for PlatformAuth & Index Read Auth
echo **** Start PW Session for PlatformAuth & Index Read Auth >> Tpm2TxtProv.log
Tpm2ProvTool StartSession EmptyAuthPwSession.sDef 1 >> Tpm2TxtProv.log

# This script requires PlatformAuth be EMPTY to create or delete NV Indexes
# Lets check if PlatformAuth is EMPTY
Tpm2ProvTool HierarchyChangeAuth PH EmptyAuth.dat 1 >> Tpm2TxtProv.log
if not %lasterror% == 0 then
	echo PlatformAuth is not EMPTY
	echo First run ResetPlatformAuth.nsh %1 %2
	goto ERROR
endif



# ***************************************************************
echo ************* Provisioning PS Index *********************
# ***************************************************************
:PSCHECK
echo **** Checking if PS Index exists
echo **** Checking if PS Index exists >> Tpm2TxtProv.log
Tpm2ProvTool NvReadPublic %2Ps%1.iDef >> Tpm2TxtProv.log
if not %lasterror% == 0 then
	echo PS Index does not exist
	goto PSCREATE		# Index does not exist 
endif
			
echo **** Comparing attributes against definition
echo **** Comparing attributes against definition >> Tpm2TxtProv.log
Tpm2ProvTool NvVerifyDef %2Ps%1.iDef >> Tpm2TxtProv.log
if %lasterror% == 0 then
	echo **** Verifying if Data is correct
	echo **** Verifying if Data is correct >> Tpm2TxtProv.log
	Tpm2ProvTool NvCompareData %2Ps%1.iDef 1 >> Tpm2TxtProv.log
	# If no error PS is correct so goto AUXCHECK
	if %lasterror% == 0 then
		goto AUXCHECK
	endif
	goto PSWRITE # Data is not correct
endif

##########################################################################
# Index attributes are incorrect
echo ****** Deleting PS Index *****
echo ****** Deleting PS Index ***** >> Tpm2TxtProv.log
echo **** Branch 1 PS Index Delete
echo **** Start Policy Session for Index auth
echo **** Start Policy Session for Index auth >> Tpm2TxtProv.log
Tpm2ProvTool StartSession Policy%1.sDef 0 >> Tpm2TxtProv.log
	
# Satisfy the PS authPolicy
# There are 4 possible policy styles
#	A. single branch with 1 assertion:    PolCommandCode(NV_UndefinespaceSpecial)
#	B. multiple branches with 1 of them:  PolCommandCode(NV_UndefinespaceSpecial)
#	C. single branch with 2 assertions:   PolOR(0,SecretValue) && PolCommandCode(NV_UndefinespaceSpecial)
#	D. multiple branches with 1 of them:  PolOR(0,SecretValue) && PolCommandCode(NV_UndefinespaceSpecial)

# Let's try A & B first
	
echo *** Policy Command Code (TPM_CC_NV_UndefineSpace)
echo *** Policy Command Code (TPM_CC_NV_UndefineSpace) >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy UnDefineSpaceSpecial.pDef 0 >> Tpm2TxtProv.log
# Tpm2ProvTool StorePolDigest 0 6 >> Tpm2TxtProv.log

# this is all that is needed for Style A (PS2) - so let's try
echo *** Execute UndefineSpecial for PS Index
echo *** Execute UndefineSpecial for PS Index >> Tpm2TxtProv.log
Tpm2ProvTool NvUndefineSpecial %2Ps%1.iDef 0 1 >> Tpm2TxtProv.log
if %lasterror% == 0 then
    goto PSCREATE
endif

# you get here if that did not work,  So lt's assume style B
#  so send the final PolicyOR assertion needed for PS1 and try again

echo *** Policy OR (Branch 1, Branch 2, Branch 3)
echo *** Policy OR (Branch 1, Branch 2, Branch 3) >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy %2PsFinalOR%1.pDef 0 >> Tpm2TxtProv.log
# Tpm2ProvTool StorePolDigest 0 6 >> Tpm2TxtProv.log

echo *** Execute UndefineSpecial for PS Index (Style B)
echo *** Execute UndefineSpecial for PS Index (Style B) >> Tpm2TxtProv.log
Tpm2ProvTool NvUndefineSpecial %2Ps%1.iDef 0 1 >> Tpm2TxtProv.log
if %lasterror% == 0 then
    goto PSCREATE
endif

# OK, Let's try style C	
# you get here if that did not work,  So lt's assume style C
# First we have to reset the Policy Session
echo **** Restart session
Tpm2ProvTool PolicyRestart 0 >> Tpm2TxtProv.log

echo *** Policy OR (0, %2PsSecret%1)
echo *** Policy OR (0, %2PsSecret%1) >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy %2SecretOR%1.pDef 0 >> Tpm2TxtProv.log
#Tpm2ProvTool StorePolDigest 0 6 >> Tpm2TxtProv.log

echo *** Policy Command Code (TPM_CC_NV_UndefineSpace)
echo *** Policy Command Code (TPM_CC_NV_UndefineSpace) >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy UnDefineSpaceSpecial.pDef 0 >> Tpm2TxtProv.log
# Tpm2ProvTool StorePolDigest 0 6 >> Tpm2TxtProv.log

# this is all that is needed for PS2 - so let's see
echo *** Execute UndefineSpecial for PS Index
Tpm2ProvTool NvUndefineSpecial %2Ps%1.iDef 0 1 >> Tpm2TxtProv.log
if %lasterror% == 0 then
    goto PSCREATE
endif

# you get here if that did not work, 
#  so send the final PolicyOR assertion needed for PS1 and try again

echo **** Policy OR (Branch 1, Branch 2, Branch 3)
echo **** Policy OR (Branch 1, Branch 2, Branch 3) >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy %2PsFinalOR%1.pDef 0 >> Tpm2TxtProv.log
Tpm2ProvTool StorePolDigest 0 6 >> Tpm2TxtProv.log

echo **** UndefineSpecial PS Index
echo **** UndefineSpecial PS Index >> Tpm2TxtProv.log
Tpm2ProvTool NvUndefineSpecial %2Ps%1.iDef 0 1 >> Tpm2TxtProv.log
if not %lasterror% == 0 then
    echo Did not satisfy index authPolicy for deleting the index
	goto ERROR		# Didn't satisfy Index authPolicy
endif

##########################################################################
:PSCREATE
echo ***** Creating PS Index *****
echo **** NV_DefineSpace for PS Index
echo **** NV_DefineSpace for PS Index >> Tpm2TxtProv.log
Tpm2ProvTool NvDefineSpace %2Ps%1.iDef 1 >> Tpm2TxtProv.log
if not %lasterror% == 0 then
    echo Requires PlatformAuth be EMPTY 
	echo If PlatformAuth is not EMPTY, then first run ResetPlatformAuth.nsh %1 %2
	goto ERROR		# Didn't create the NV Index
endif

##########################################################################
:PSWRITE
echo ***** Writing PS  *****
echo ***** Writing PS  ***** >> Tpm2TxtProv.log

echo **** Start Policy Session
echo **** Start Policy Session >> Tpm2TxtProv.log
Tpm2ProvTool StartSession Policy%1.sDef 0 >> Tpm2TxtProv.log
# There are two branches (2 & 3) that can be used to write a new PS Index
# The NvNotWritten branch can only be used on a newly created index
# but not for re-write. However, it is the only branch that works for DEFAULT 
 
 if NOT EXIST %2SecretOR%1.pDef then  
	echo Using NV-Not-Written branch to write PS
	goto NOTWRITTEN
endif

# Satisfy Branch 2
echo **** Policy Branch 2
echo **** Policy Branch 2 >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy %2SecretOR%1.pDef 0 >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy %2PsFinalOR%1.pDef 0 >> Tpm2TxtProv.log

echo **** Writing NV Data
echo **** Writing NV Data >> Tpm2TxtProv.log
Tpm2ProvTool NvWrite %2Ps%1.iDef 0 >> Tpm2TxtProv.log
if %lasterror% == 0 then
	goto AUXCHECK		# Done with PS
endif

# Lets try the NvNotWritten Branch
:NOTWRITTEN
Tpm2ProvTool PolicyRestart 0 >> Tpm2TxtProv.log
# Satisfy Branch 3
echo **** Policy Branch 3
echo **** Policy Branch 3 >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy NvNotWritten.pDef 0 >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy %2PsFinalOR%1.pDef 0 >> Tpm2TxtProv.log

echo **** Writing NV Data
echo **** Writing NV Data >> Tpm2TxtProv.log
Tpm2ProvTool NvWrite %2Ps%1.iDef 0 >> Tpm2TxtProv.log
if not %lasterror% == 0 then
	goto ERROR		# Didn't satisfy Index authPolicy
endif

goto AUXCHECK
goto PSCHECK # for testing on NEX

##########################################################################
# ***************************************************************
# *************************AUX INDEX ****************************
# ***************************************************************
:AUXCHECK
echo **** Checking AUX Index

echo **** Checking if AUX index exists
echo **** Checking if AUX index exists >> Tpm2TxtProv.log
Tpm2ProvTool NvReadPublic Aux%3%1.iDef >> Tpm2TxtProv.log
if not %lasterror% == 0 then
	echo Aux Index does not exist
	goto AUXCREATE		# Index does not exist 
endif

echo **** AUX already exists, check if provisioned correctly 
echo **** AUX already exists, check if provisioned correctly  >> Tpm2TxtProv.log
Tpm2ProvTool NvVerifyDef Aux%3%1.iDef >> Tpm2TxtProv.log
if %lasterror% == 0 then 
	echo Aux Index provisioned correctly
	goto DONE
endif

##########################################################################
echo **** Need to delete AUX Index
echo **** Need to delete AUX Index >> Tpm2TxtProv.log
 
 if NOT EXIST %2SecretOR%1.pDef then  
	echo PS index data must be modified to allow AUX deletion
	echo Cannot write PS Index -- Default indexes cannot be modified
	goto ERROR
endif

echo **** Rewriting PS Policy to set Aux-Delete-Enable
echo **** Rewriting PS Policy to set Aux-Delete-Enable >> Tpm2TxtProv.log

:PSREWRITE1
echo **** Start Policy Session to Write PS
echo **** Start Policy Session >> Tpm2TxtProv.log
Tpm2ProvTool StartSession Policy%1.sDef 0 >> Tpm2TxtProv.log
# Satisfy Branch 2
echo **** Policy Branch 2
echo **** Policy Branch 2 >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy %2SecretOR%1.pDef 0 >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy %2PsFinalOR%1.pDef 0 >> Tpm2TxtProv.log

echo **** Writing PS NV Data
echo **** Writing PS NV Data >> Tpm2TxtProv.log
Tpm2ProvTool NvWrite PsDelAux%3.iDef 0 >> Tpm2TxtProv.log
if not %lasterror% == 0 then
	goto ERROR		# Didn't satisfy Index authPolicy
endif

:AUXDELETE
#echo ***** Deleting AUX Index *****
echo **** "ServerGetSec -l ent -fn res"
echo **** "ServerGetSec -l ent -fn res" >> Tpm2TxtProv.log
getsec64.efi -l ent -fn res >> Tpm2TxtProv.log
if not %lasterror% == 0 then
	echo Unable to delete Aux index
	goto ERROR		# Didn't delete Index 
endif
goto PSREWRITE2
##########################################################################
# Delete AUX on Simulator or TTV
echo **** Start Policy Session 
Tpm2ProvTool StartSession Policy%1.sDef 0  # For testing on NEX
echo **** Locality3&4 Assertion 
Tpm2ProvTool AssertPolicy Locality3&4.pDef 0  # For testing on NEX
echo **** CommandCode assertion
Tpm2ProvTool AssertPolicy UndefineSpaceSpecial.pDef 0  # For testing on NEX
echo **** PolicyOR assertion
Tpm2ProvTool AssertPolicy AuxPolicyOR%1.pDef 0  # For testing on NEX
echo **** change to locality 3 
Tpm2ProvTool SetActiveLocality 3 # For testing on NEX
echo **** UndefineSpaceSpecial for AUX
Tpm2ProvTool NvUndefineSpecial Aux%3%1.iDef 0 1  # For testing on NEX
if not %lasterror% == 0 then
    Tpm2ProvTool SetActiveLocality 0
	echo Cannot delete AUX index
	set FAIL = TRUE
	goto ERROR		
endif
Tpm2ProvTool SetActiveLocality 0 # For testing on NEX

##########################################################################
:PSREWRITE2
echo *** Start Policy Session to re-write PS
echo *** Start Policy Session to re-write PS >> Tpm2TxtProv.log
Tpm2ProvTool StartSession Policy%1.sDef 0 >> Tpm2TxtProv.log
# Satisfy Branch 2
echo **** Policy Branch 2
echo **** Policy Branch 2 >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy %2SecretOR%1.pDef 0 >> Tpm2TxtProv.log
Tpm2ProvTool AssertPolicy %2PsFinalOR%1.pDef 0 >> Tpm2TxtProv.log

echo **** Writing NV Data
echo **** Writing PS NV Data >> Tpm2TxtProv.log
Tpm2ProvTool NvWrite %2Ps%1.iDef 0 >> Tpm2TxtProv.log
if not %lasterror% == 0 then
	echo Unable to rewrite PS Index
	goto ERROR		# Didn't write Index 
endif
#
# Need to check to make sure AUX Deletion was really successful
echo **** Checking if AUX index was deleted
echo **** Checking if AUX index was deleted >> Tpm2TxtProv.log
Tpm2ProvTool NvReadPublic Aux%3%1.iDef >> Tpm2TxtProv.log
if %lasterror% == 0 then
	echo Aux Index was not deleted
	goto ERROR		# Index still exists
endif
##########################################################################
:AUXCREATE
echo ******** Creating Aux Index ********
echo **** AUX NV_DefineSpace
echo **** AUX NV_DefineSpace >> Tpm2TxtProv.log
Tpm2ProvTool NvDefineSpace Aux%3%1.iDef 1 >> Tpm2TxtProv.log
if not %lasterror% == 0 then
	echo Unable to create AUX Index
	goto ERROR		
endif
goto DONE
goto AUXCHECK # for testing on NEX
##########################################################################


:DONE
echo  ***************************************************
echo  ******* Provisioning Completed Successfully *******
echo  ***************************************************

goto EOF

:ERROR
echo  ***************************************************
echo  *************** Provisioning FAILED ***************
echo  ***************************************************
Tpm2ProvTool FlushSession 0 >> Tpm2TxtProv.log

goto EOF

:EOF



